"use client";

import { useState, useEffect } from "react";
import { useRouter } from "next/navigation";
import { Card, CardContent } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Badge } from "@/components/ui/badge";
import { Alert, AlertDescription } from "@/components/ui/alert";
import {
  Bot,
  Wallet,
  MessageCircle,
  BarChart3,
  Play,
  LogOut,
  Bell,
  Settings,
  User,
  CreditCard,
  Activity,
  AlertCircle,
  RefreshCw,
  Brain,
} from "lucide-react";
import { Avatar, AvatarFallback, AvatarImage } from "@/components/ui/avatar";
import {
  DropdownMenu,
  DropdownMenuContent,
  DropdownMenuItem,
  DropdownMenuLabel,
  DropdownMenuSeparator,
  DropdownMenuTrigger,
} from "@/components/ui/dropdown-menu";
import Image from "next/image";
import { PageTransition } from "@/components/page-transition";
import { SessionExpiredModal } from "@/components/session-expired-modal";
import { useSessionExpired } from "@/hooks/use-session-expired";

interface UserData {
  id?: number;
  login?: string;
  credit?: string;
  credit_cents?: number;
  freebet?: string;
  freebet_cents?: number;
  bonus?: string;
  bonus_cents?: number;
  type?: number;
  last_login?: string;
}

export default function DashboardPage() {
  const [userData, setUserData] = useState<UserData | null>(null);
  const [isLoading, setIsLoading] = useState(true);
  const [error, setError] = useState<string | null>(null);
  const router = useRouter();
  const {
    sessionExpiredOpen,
    sessionExpiredMessage,
    triggerSessionExpired,
    handleSessionExpiredRedirect,
  } = useSessionExpired();

  const fetchUserData = async () => {
    try {
      const token = localStorage.getItem("token");
      const storedUser = localStorage.getItem("user");
      const storedBalance = localStorage.getItem("userBalance");
      let parsedUser: any = null;
      if (storedUser) {
        try {
          parsedUser = JSON.parse(storedUser);
        } catch {
          parsedUser = null;
        }
      }

      if (!token) {
        triggerSessionExpired();
        return;
      }

      // Usa cache local imediatamente, se existir
      if (parsedUser && storedBalance) {
        setUserData({ ...parsedUser, credit: storedBalance });
        setError(null);
        setIsLoading(false);
        return;
      }

      const userIdValue = parsedUser?.id ?? parsedUser?.usuario?.id;
      if (userIdValue === undefined || userIdValue === null) {
        triggerSessionExpired(
          "Não foi possível validar sua sessão. Faça login novamente.",
        );
        return;
      }

      const userId = String(userIdValue);

      // Busca SEMPRE via proxy interno (backend separado do frontend)
      const response = await fetch(
        `/api/user/balance?id=${encodeURIComponent(userId)}`,
        {
          method: "GET",
          headers: {
            Authorization: `Bearer ${token}`,
            "Content-Type": "application/json",
          },
          cache: "no-store",
        },
      );

      const data = await response.json().catch(() => ({}));

      if (response.ok) {
        setUserData(data);
        setError(null);
        // persiste user
        localStorage.setItem("user", JSON.stringify(data));

        // >>> SINCRONIZAÇÃO DE SALDO (sempre salva como string "10.000,00"):
        if (data?.credit !== undefined) {
          const formatted =
            typeof data.credit === "string"
              ? data.credit
              : new Intl.NumberFormat("pt-BR", {
                  minimumFractionDigits: 2,
                  maximumFractionDigits: 2,
                }).format(Number(data.credit) / 100);
          localStorage.setItem("userBalance", formatted);
        } else if (typeof data?.credit_cents === "number") {
          const formatted = new Intl.NumberFormat("pt-BR", {
            minimumFractionDigits: 2,
            maximumFractionDigits: 2,
          }).format(data.credit_cents / 100);
          localStorage.setItem("userBalance", formatted);
        }
      } else {
        setError(data?.message || "Erro ao carregar dados");

        if (response.status === 401) {
          triggerSessionExpired(data?.message);
        }
      }
    } catch {
      setError("Erro de conexão com o servidor");
    } finally {
      setIsLoading(false);
    }
  };

  useEffect(() => {
    fetchUserData();
    // eslint-disable-next-line react-hooks/exhaustive-deps
  }, [router]);

  // >>> ESCUTA ALTERAÇÕES DO SALDO FEITAS EM OUTRA PÁGINA/ABA:
  useEffect(() => {
    const onStorage = (e: StorageEvent) => {
      if (e.key === "userBalance" && e.newValue) {
        setUserData((prev) =>
          prev
            ? { ...prev, credit: e.newValue! }
            : ({ credit: e.newValue! } as UserData),
        );
      }
    };
    window.addEventListener("storage", onStorage);
    return () => window.removeEventListener("storage", onStorage);
  }, []);

  const handleLogout = () => {
    localStorage.removeItem("token");
    localStorage.removeItem("user");
    localStorage.removeItem("userBalance");
    router.push("/login");
  };

  const handleRefresh = async () => {
    setIsLoading(true);
    setError(null);
    localStorage.removeItem("user");
    localStorage.removeItem("userBalance");
    await fetchUserData();
  };

  // Formata saldo vindo como string já formatada (ex: "7.987,20")
  // ou como centavos (credit_cents)
  const formatCurrency = (value: string | number | undefined) => {
    if (value === undefined || value === null) return "R$ 0,00";

    if (typeof value === "string") {
      // já vem formatado do provedor
      return `R$ ${value}`;
    }

    // número em centavos
    return new Intl.NumberFormat("pt-BR", {
      style: "currency",
      currency: "BRL",
    }).format(Number(value) / 100);
  };

  const formatLastLogin = (lastLogin: string | undefined) => {
    if (!lastLogin) return "";
    try {
      const date = new Date(lastLogin);
      return date.toLocaleString("pt-BR");
    } catch {
      return lastLogin;
    }
  };

  if (isLoading) {
    return (
      <>
        <PageTransition>
          <div className="min-h-screen bg-gradient-to-br from-gray-950 via-slate-900 to-gray-950 flex items-center justify-center">
            <div className="text-center">
              <div className="animate-spin rounded-full h-32 w-32 border-b-2 border-blue-500 mx-auto mb-4"></div>
              <p className="text-white text-lg">Carregando seus dados...</p>
            </div>
          </div>
        </PageTransition>
        <SessionExpiredModal
          open={sessionExpiredOpen}
          message={sessionExpiredMessage}
          onConfirm={handleSessionExpiredRedirect}
        />
      </>
    );
  }

  if (error && !userData) {
    return (
      <>
        <PageTransition>
          <div className="min-h-screen bg-gradient-to-br from-gray-950 via-slate-900 to-gray-950 flex items-center justify-center">
            <div className="w-full max-w-md p-6 bg-gray-900/80 rounded-xl border border-gray-800 backdrop-blur-xl">
              <div className="text-center mb-6">
                <AlertCircle className="h-12 w-12 text-red-400 mx-auto mb-4" />
                <h2 className="text-2xl font-bold text-white mb-2">
                  Erro de Conexão
                </h2>
                <p className="text-gray-400 mb-4">{error}</p>
              </div>

              <Button
                className="w-full bg-gradient-to-r from-blue-600 to-cyan-500 hover:from-blue-700 hover:to-cyan-600 mb-3"
                onClick={handleRefresh}
              >
                <RefreshCw className="mr-2 h-4 w-4" />
                Tentar Novamente
              </Button>

              <Button
                variant="outline"
                className="w-full border-gray-700 text-gray-300 hover:bg-gray-800 bg-transparent"
                onClick={() => router.push("/login")}
              >
                Voltar para o Login
              </Button>
            </div>
          </div>
        </PageTransition>
        <SessionExpiredModal
          open={sessionExpiredOpen}
          message={sessionExpiredMessage}
          onConfirm={handleSessionExpiredRedirect}
        />
      </>
    );
  }

  return (
    <>
      <PageTransition>
        <div className="min-h-screen bg-gradient-to-br from-gray-950 via-slate-900 to-gray-950">
          {/* Background Effects */}
          <div className="absolute inset-0 overflow-hidden">
            <div className="absolute -top-40 -right-40 w-80 h-80 bg-blue-600/5 rounded-full mix-blend-multiply filter blur-xl animate-pulse"></div>
            <div className="absolute -bottom-40 -left-40 w-80 h-80 bg-cyan-500/5 rounded-full mix-blend-multiply filter blur-xl animate-pulse delay-1000"></div>
          </div>

          <div className="relative z-10">
            {/* Header */}
            <header className="border-b border-gray-800/50 bg-gray-900/30 backdrop-blur-xl">
              <div className="container mx-auto px-4 py-4">
                <div className="flex items-center justify-between">
                  <div className="flex items-center space-x-4">
                    <div className="flex items-center space-x-3">
                      <div className="p-2 bg-gray-800/50 rounded-lg">
                        <Image
                          src="/logo-optimus.png"
                          alt="OptimusIA Logo"
                          width={32}
                          height={32}
                          className="drop-shadow-lg"
                        />
                      </div>
                      <span className="text-xl font-bold text-white">
                        OptimusIA
                      </span>
                    </div>
                  </div>

                  <div className="flex items-center space-x-4">
                    <Button
                      variant="ghost"
                      size="icon"
                      className="text-gray-400 hover:text-white hover:bg-gray-800"
                      onClick={handleRefresh}
                    >
                      <RefreshCw className="w-5 h-5" />
                    </Button>

                    <Button
                      variant="ghost"
                      size="icon"
                      className="text-gray-400 hover:text-white hover:bg-gray-800"
                    >
                      <Bell className="w-5 h-5" />
                    </Button>

                    <DropdownMenu>
                      <DropdownMenuTrigger asChild>
                        <Button
                          variant="ghost"
                          className="relative h-8 w-8 rounded-full"
                        >
                          <Avatar className="h-8 w-8">
                            <AvatarImage
                              src="/placeholder.svg?height=32&width=32"
                              alt="Avatar"
                            />
                            <AvatarFallback className="bg-gradient-to-r from-blue-600 to-cyan-500 text-white">
                              {userData?.login?.charAt(0).toUpperCase() || "U"}
                            </AvatarFallback>
                          </Avatar>
                        </Button>
                      </DropdownMenuTrigger>
                      <DropdownMenuContent
                        className="w-56 bg-gray-900 border-gray-800"
                        align="end"
                        forceMount
                      >
                        <DropdownMenuLabel className="font-normal">
                          <div className="flex flex-col space-y-1">
                            <p className="text-sm font-medium leading-none text-white">
                              {userData?.login || "Usuário"}
                            </p>
                            <p className="text-xs leading-none text-gray-400">
                              ID: {userData?.id}
                            </p>
                            {userData?.last_login && (
                              <p className="text-xs leading-none text-gray-400">
                                Último login:{" "}
                                {formatLastLogin(userData.last_login)}
                              </p>
                            )}
                          </div>
                        </DropdownMenuLabel>
                        <DropdownMenuSeparator className="bg-gray-800" />
                        <DropdownMenuItem className="text-gray-300 hover:text-white hover:bg-gray-800">
                          <User className="mr-2 h-4 w-4" />
                          <span>Perfil</span>
                        </DropdownMenuItem>
                        <DropdownMenuItem className="text-gray-300 hover:text-white hover:bg-gray-800">
                          <Settings className="mr-2 h-4 w-4" />
                          <span>Configurações</span>
                        </DropdownMenuItem>
                        <DropdownMenuSeparator className="bg-gray-800" />
                        <DropdownMenuItem
                          className="text-red-400 hover:text-red-300 hover:bg-gray-800"
                          onClick={handleLogout}
                        >
                          <LogOut className="mr-2 h-4 w-4" />
                          <span>Sair</span>
                        </DropdownMenuItem>
                      </DropdownMenuContent>
                    </DropdownMenu>
                  </div>
                </div>
              </div>
            </header>

            {/* Main Content */}
            <main className="container mx-auto px-4 py-8">
              {/* Error Alert (não bloqueante) */}
              {error && userData && (
                <Alert className="mb-6 bg-yellow-500/10 border-yellow-500/30 text-yellow-400">
                  <AlertCircle className="h-4 w-4" />
                  <AlertDescription>{error}</AlertDescription>
                </Alert>
              )}

              {/* Welcome Section */}
              <div className="mb-8">
                <div className="flex flex-col md:flex-row md:items-center md:justify-between mb-6">
                  <div>
                    <h1 className="text-3xl font-bold text-white mb-2">
                      Seja bem-vindo, {userData?.login || "Trader"}! 🤖
                    </h1>
                    <p className="text-gray-400">
                      Pronto para operar com inteligência artificial?
                    </p>
                  </div>
                  <div className="mt-4 md:mt-0 flex items-center space-x-2">
                    <Brain className="w-6 h-6 text-blue-400" />
                    <Badge
                      variant="secondary"
                      className="bg-blue-600/20 text-blue-400 border-blue-600/30"
                    >
                      IA Ativa
                    </Badge>
                  </div>
                </div>

                {/* Balance Card */}
                <Card className="bg-gradient-to-red from-green-600/20 to-emerald-600/20 border-green-600/30 backdrop-blur-xl shadow-xl">
                  <CardContent className="p-6">
                    <div className="flex items-center justify-between">
                      <div>
                        <p className="text-green-400 text-sm font-medium mb-1">
                          Saldo Disponível
                        </p>
                        <p className="text-3xl font-bold text-white drop-shadow-lg">
                          {formatCurrency(
                            userData?.credit ?? userData?.credit_cents,
                          )}
                        </p>
                      </div>
                      <div className="w-12 h-12 bg-green-600/30 rounded-full flex items-center justify-center">
                        <Wallet className="w-6 h-6 text-green-400" />
                      </div>
                    </div>
                    <div className="mt-4 flex items-center space-x-4 text-sm">
                      <div className="flex items-center space-x-1 text-green-400">
                        <Activity className="w-4 h-4" />
                      </div>
                      <div className="text-gray-400">
                        Última atualização: agora
                      </div>
                    </div>
                  </CardContent>
                </Card>
              </div>

              {/* Action Cards Grid */}
              <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-2 gap-6 mb-8">
                <Card className="bg-gray-900/50 border-gray-800 backdrop-blur-xl hover:bg-gray-900/70 transition-all duration-300 cursor-pointer group shadow-lg">
                  <CardContent className="p-6">
                    <div className="flex items-center justify-between mb-4">
                      <div className="w-12 h-12 bg-blue-600/20 rounded-xl flex items-center justify-center group-hover:scale-110 transition-transform">
                        <CreditCard className="w-6 h-6 text-blue-400" />
                      </div>
                      <div className="text-right">
                        <p className="text-2xl font-bold text-white">
                          Depositar
                        </p>
                        <p className="text-gray-400 text-sm">
                          Adicionar fundos
                        </p>
                      </div>
                    </div>
                    <p className="text-gray-300 text-sm">
                      Faça depósitos instantâneos via PIX, TED ou cartão de
                      crédito
                    </p>
                  </CardContent>
                </Card>

                <Card className="bg-gray-900/50 border-gray-800 backdrop-blur-xl hover:bg-gray-900/70 transition-all duration-300 cursor-pointer group shadow-lg">
                  <CardContent className="p-6">
                    <div className="flex items-center justify-between mb-4">
                      <div className="w-12 h-12 bg-purple-600/20 rounded-xl flex items-center justify-center group-hover:scale-110 transition-transform">
                        <Bot className="w-6 h-6 text-purple-400" />
                      </div>
                      <div className="text-right">
                        <p className="text-2xl font-bold text-white">
                          IA Assistant
                        </p>
                        <p className="text-gray-400 text-sm">
                          Análises inteligentes
                        </p>
                      </div>
                    </div>
                    <p className="text-gray-300 text-sm">
                      Receba análises e sinais gerados por inteligência
                      artificial
                    </p>
                  </CardContent>
                </Card>

                <Card
                  className="bg-gray-900/50 border-gray-800 backdrop-blur-xl hover:bg-gray-900/70 transition-all duration-300 cursor-pointer group shadow-lg"
                  onClick={() =>
                    window.open(
                      "https://chat.whatsapp.com/LI2KLANTfde9x3fyNgNs7m?mode=ems_copy_c",
                      "_blank",
                    )
                  }
                >
                  <CardContent className="p-6">
                    <div className="flex items-center justify-between mb-4">
                      <div className="w-12 h-12 bg-cyan-600/20 rounded-xl flex items-center justify-center group-hover:scale-110 transition-transform">
                        <MessageCircle className="w-6 h-6 text-cyan-400" />
                      </div>
                      <div className="text-right">
                        <p className="text-2xl font-bold text-white">
                          Comunidade
                        </p>
                        <p className="text-gray-400 text-sm">Telegram</p>
                      </div>
                    </div>
                    <p className="text-gray-300 text-sm">
                      Junte-se ao nosso grupo para dicas e análises diárias
                    </p>
                  </CardContent>
                </Card>

                <Card className="bg-gray-900/50 border-gray-800 backdrop-blur-xl hover:bg-gray-900/70 transition-all duration-300 cursor-pointer group shadow-lg">
                  <CardContent className="p-6">
                    <div className="flex items-center justify-between mb-4">
                      <div className="w-12 h-12 bg-orange-600/20 rounded-xl flex items-center justify-center group-hover:scale-110 transition-transform">
                        <BarChart3 className="w-6 h-6 text-orange-400" />
                      </div>
                      <div className="text-right">
                        <p className="text-2xl font-bold text-white">
                          Estatísticas
                        </p>
                        <p className="text-gray-400 text-sm">Seus resultados</p>
                      </div>
                    </div>
                    <p className="text-gray-300 text-sm">
                      Acompanhe seu desempenho e histórico de operações
                    </p>
                  </CardContent>
                </Card>
              </div>

              {/* Start Trading Button */}
              <div className="text-center">
                <Button
                  size="lg"
                  className="bg-gradient-to-r from-blue-600 to-cyan-500 hover:from-blue-700 hover:to-cyan-600 text-white font-semibold px-12 py-4 text-lg rounded-xl shadow-lg hover:shadow-blue-500/25 transition-all duration-300"
                  onClick={() => router.push("/ai-operation")}
                >
                  <Play className="mr-2 h-5 w-5" />
                  Iniciar Operação com IA
                </Button>
                <p className="text-gray-400 text-sm mt-3">
                  Comece a operar com o poder da inteligência artificial
                </p>
              </div>
            </main>
          </div>
        </div>
      </PageTransition>
      <SessionExpiredModal
        open={sessionExpiredOpen}
        message={sessionExpiredMessage}
        onConfirm={handleSessionExpiredRedirect}
      />
    </>
  );
}
